unit MainFormU;

{$DEFINE USE_CUSTOM_MSG_BOX}

interface

uses
  System.SysUtils, System.Types, System.UITypes, System.Classes, System.Variants,
  FMX.Types, FMX.Controls, FMX.Forms, FMX.Graphics, FMX.Dialogs,
  Androidapi.JNI.GraphicsContentViewText, FMX.ListBox, FMX.Edit, FMX.StdCtrls,
  FMX.Layouts;

type
  TMainForm = class(TForm)
    StyleBookLight: TStyleBook;
    StyleBookDark: TStyleBook;
    ListBox1: TListBox;
    Layout1: TLayout;
    Button1: TButton;
    Edit1: TEdit;
    Edit2: TEdit;
    Edit3: TEdit;
    Edit4: TEdit;
    ListBoxItem1: TListBoxItem;
    ListBoxItem2: TListBoxItem;
    ListBoxItem3: TListBoxItem;
    procedure FormCreate(Sender: TObject);
    procedure Button1Click(Sender: TObject);
  private
    { Private declarations }
    const ScanRequestCode = 0;
    procedure RegisterDelphiNativeMethods;
    function GetResourceId(const ResourceName: string): Integer;
    function OnOptionsItemSelected(MenuItem: JMenuItem): Boolean;
    function OnActivityResult(RequestCode, ResultCode: Integer; Data: JIntent): Boolean;
  public
    { Public declarations }
  end;

var
  MainForm: TMainForm;

implementation

uses
  Posix.pthread,
  FMX.Helpers.Android,
  Androidapi.NativeActivity,
  Androidapi.JNI,
  Androidapi.JNIBridge,
  Androidapi.JNI.JavaTypes,
  Androidapi.JNI.App, LaunchIntents;

{$R *.fmx}

{$REGION 'JNI setup code and callback'}
var
  ARNRequestCode, ARNResultCode: Integer;
  ARNData: JIntent;
  ARNMenuItem: JMenuItem;
  ARNResult: Boolean;

procedure OnActivityResultThreadSwitcher;
begin
  Log.d('+OnActivityResultThreadSwitcher');
  Log.d('Thread (Main: %.8x, Current: %.8x, Java:%.8d (%2:.8x), POSIX:%.8x)',
    [MainThreadID, TThread.CurrentThread.ThreadID, TJThread.JavaClass.currentThread.getId, GetCurrentThreadID]);
  ARNResult := MainForm.OnActivityResult(ARNRequestCode, ARNResultCode, ARNData);
  Log.d('-OnActivityResultThreadSwitcher');
end;

procedure OnOptionsItemSelectedThreadSwitcher;
begin
  Log.d('+OnOptionsItemSelectedThreadSwitcher');
  Log.d('Thread (Main: %.8x, Current: %.8x, Java:%.8d (%2:.8x), POSIX:%.8x)',
    [MainThreadID, TThread.CurrentThread.ThreadID, TJThread.JavaClass.currentThread.getId, GetCurrentThreadID]);
  ARNResult := MainForm.OnOptionsItemSelected(ARNMenuItem);
  Log.d('-OnOptionsItemSelectedThreadSwitcher');
end;

//This is called from the Java activity's onActivityResult() method
function OnActivityResultNative(PEnv: PJNIEnv; This: JNIObject;
  RequestCode, ResultCode: Integer; dataIntent: JNIObject): Boolean; cdecl;
begin
  Log.d('+OnActivityResultNative');
  Log.d('Thread (Main: %.8x, Current: %.8x, Java:%.8d (%2:.8x), POSIX:%.8x)',
    [MainThreadID, TThread.CurrentThread.ThreadID, TJThread.JavaClass.currentThread.getId, GetCurrentThreadID]);
  ARNRequestCode := requestCode;
  ARNResultCode := resultCode;
  ARNData := nil;
  if Assigned(DataIntent) then
    ARNData := TJIntent.Wrap(DataIntent);
  if DataIntent = nil then
    Log.d(Format('OnActivityResultNative(%d,%d,nil)',
      [requestCode, resultCode]))
  else
    Log.d(Format('OnActivityResultNative(%d,%d,%s)',
      [requestCode, resultCode, JStringToString(ARNData.ToString)]));
  // Pass on the activity result to the helper for handling
  Log.d('Calling Synchronize');
  TThread.Synchronize(nil, OnActivityResultThreadSwitcher);
  Log.d('Synchronize is over');
  Result := ARNResult;
  Log.d('-OnActivityResultNative');
end;

//This is called from the Java activity's onOptionsItemSelected() method
function OnOptionsItemSelectedNative(PEnv: PJNIEnv; This: JNIObject; JNIMenuItem: JNIObject): Boolean; cdecl;
begin
  Log.d('+OnOptionsItemSelectedNative');
  Log.d('Thread (Main: %.8x, Current: %.8x, Java:%.8d (%2:.8x), POSIX:%.8x)',
    [MainThreadID, TThread.CurrentThread.ThreadID, TJThread.JavaClass.currentThread.getId, GetCurrentThreadID]);
  ARNMenuItem := TJMenuItem.Wrap(JNIMenuItem);
  Log.d('Calling Synchronize');
  TThread.Synchronize(nil, OnOptionsItemSelectedThreadSwitcher);
  Log.d('Synchronize is over');
  Result := ARNResult;
  Log.d('-OnOptionsItemSelectedNative');
end;

procedure TMainForm.RegisterDelphiNativeMethods;
var
  PEnv: PJNIEnv;
  ActivityClass: JNIClass;
  NativeMethods: array[0..1] of JNINativeMethod;
begin
  Log.d('Starting the registration JNI stuff');

  PEnv := TJNIResolver.GetJNIEnv;

  Log.d('Registering interop methods');

  NativeMethods[0].Name := 'onActivityResultNative';
  NativeMethods[0].Signature := '(IILandroid/content/Intent;)Z';
  NativeMethods[0].FnPtr := @OnActivityResultNative;

  NativeMethods[1].Name := 'onOptionsItemSelectedNative';
  NativeMethods[1].Signature := '(Landroid/view/MenuItem;)Z';
  NativeMethods[1].FnPtr := @OnOptionsItemSelectedNative;

  ActivityClass := PEnv^.GetObjectClass(
    PEnv, PANativeActivity(System.DelphiActivity).clazz);

  PEnv^.RegisterNatives(PEnv, ActivityClass, @NativeMethods[0], 2);

  PEnv^.DeleteLocalRef(PEnv, ActivityClass);
end;
{$ENDREGION}

{$REGION 'Message box wrappers'}
{$IF Defined(ANDROID) and Defined(USE_CUSTOM_MSG_BOX)}
//There is a bug logged internally regarding the FMX messgae box routines
//(ShowMessage & MessageDlg) hanging when called from mobile OS callbacks
//Use of custom message boxes (say implemented in Java) avoids this, but
//in this example they are non-blocking
procedure NativeShowMsg(const Title, Msg: String);
var
  PActivity: PANativeActivity;
  PEnv: PJNIEnv;
  ActivityClass: JNIClass;
  GetMethod: JNIMethodID;
begin
  PActivity := PANativeActivity(System.DelphiActivity);
  PEnv := TJNIResolver.GetJNIEnv;
  ActivityClass := PEnv^.GetObjectClass(PEnv, PActivity^.clazz);
  GetMethod := TJNIResolver.GetJavaMethodID(ActivityClass,
    'showDialog', '(Ljava/lang/String;Ljava/lang/String;)V');
  PEnv^.CallVoidMethodA(PEnv, PActivity^.clazz, GetMethod,
    PJNIValue(ArgsToJNIValues([StringToJNIString(PEnv, Title),
                               StringToJNIString(PEnv, Msg)])));
  PEnv^.DeleteLocalRef(PEnv, ActivityClass);
end;

procedure ShowMessage(const Msg: String);
begin
  CallInUIThread(
  procedure
  begin
    NativeShowMsg(Application.Title, Msg)
  end);
end;

function MessageDlg(const Msg: string; DlgType: TMsgDlgType;
  Buttons: TMsgDlgButtons; HelpCtx: Integer): Integer;
var
  Title: string;
begin
  CallInUIThread(
  procedure
  begin
    case DlgType of
      TMsgDlgType.mtInformation: Title := 'Information';
      TMsgDlgType.mtWarning: Title := 'Warning';
      TMsgDlgType.mtError: Title := 'Error';
    else
      Title := Application.Title
    end;
    NativeShowMsg(Title, Msg);
  end);
  Result := 0;
end;
{$ENDIF}
{$ENDREGION}

{ TMainForm }

procedure TMainForm.FormCreate(Sender: TObject);
begin
  RegisterDelphiNativeMethods
end;

procedure TMainForm.Button1Click(Sender: TObject);
begin
  FMX.Dialogs.ShowMessage('Starting bar code scan');
  LaunchQRScanner(ScanRequestCode);
end;

function TMainForm.OnActivityResult(RequestCode, ResultCode: Integer;
  Data: JIntent): Boolean;
var
  ScanContent,
  ScanFormat: string;
begin
  Log.d('+TMainForm.OnActivityResult');
  Log.d('Thread (Main: %.8x, Current: %.8x, Java:%.8d (%2:.8x), POSIX:%.8x)',
    [MainThreadID, TThread.CurrentThread.ThreadID,
     TJThread.JavaClass.currentThread.getId, GetCurrentThreadID]);

  Result := False;
  //For more info see http://code.google.com/p/zxing/wiki/ScanningViaIntent

  if RequestCode = ScanRequestCode then
  begin
    if ResultCode = TJActivity.JavaClass.RESULT_OK then
    begin
      if Assigned(Data) then
      begin
        ScanContent := JStringToString(Data.getStringExtra(StringToJString('SCAN_RESULT')));
        ScanFormat := JStringToString(Data.getStringExtra(StringToJString('SCAN_RESULT_FORMAT')));
        ListBox1.Items.Add(Format('%s code: %s', [ScanFormat, ScanContent]));
        ShowMessage(Format('Content = %s'#10#10'Format = %s', [ScanContent, ScanFormat]))
      end;
    end
    else if ResultCode = TJActivity.JavaClass.RESULT_CANCELED then
    begin
      ShowMessage('You cancelled the scan....')
    end;
    Result := True;
  end;

  Log.d('-TMainForm.OnActivityResult');
end;

function TMainForm.GetResourceId(const ResourceName: string): Integer;
begin
  Result := SharedActivity.getResources.getIdentifier(
    StringToJString(ResourceName),
    nil,
    SharedActivity.getPackageName)
end;

function TMainForm.OnOptionsItemSelected(MenuItem: JMenuItem): Boolean;
begin
  Log.d('+TMainForm.OnOptionsItemSelected');
  Log.d('Thread (Main: %.8x, Current: %.8x, Java:%.8d (%2:.8x), POSIX:%.8x)',
    [MainThreadID, TThread.CurrentThread.ThreadID,
     TJThread.JavaClass.currentThread.getId, GetCurrentThreadID]);

  if MenuItem.getItemId = GetResourceId('id/menu_dark_theme') then
  begin
    Self.StyleBook := StyleBookDark;
    Result := True;
  end
  else
  if MenuItem.getItemId = GetResourceId('id/menu_light_theme') then
  begin
    Self.StyleBook := StyleBookLight;
    Result := True;
  end
  else
    Result := False;

  Log.d('-TMainForm.OnOptionsItemSelected');
end;

end.
