unit Delphi.Platform.iOS;

interface

uses
  System.TypInfo,
  Macapi.ObjectiveC,
  iOSapi.Foundation,
  iOSapi.UIKit;

type
{$M+}
  id = Pointer;
  SEL = Pointer;

  TApplicationDelegate = class;

  TCocoaTouch = class(TInterfacedObject)
  private
    FApplication: UIApplication;
    FAppDelegate: TApplicationDelegate;
    FMainScreen: UIScreen;
    FCurrentDevice: UIDevice;
  public
    constructor Create;
    procedure Run;
    //Interface reference to iOS UIApplication object
    property Application: UIApplication read FApplication;
    //Object reference to application delegate
    property AppDelegate: TApplicationDelegate read FAppDelegate;
    property CurrentDevice: UIDevice read FCurrentDevice;
    property MainScreen: UIScreen read FMainScreen;
  end;

  TApplicationDelegate = class{(TOCLocal, UIApplicationDelegate)}
  public
    constructor Create;
    destructor Destroy; override;
    function application(const Sender: UIApplication; const didFinishLaunchingWithOptions: NSDictionary): Boolean; overload; virtual; cdecl;
    procedure applicationDidBecomeActive(const Sender: UIApplication); virtual; cdecl;
    procedure applicationDidEnterBackground(const Sender: UIApplication); virtual; cdecl;
    procedure applicationDidReceiveMemoryWarning(const Sender: UIApplication); virtual; cdecl;
    procedure applicationWillEnterForeground(const Sender: UIApplication); virtual; cdecl;
    procedure applicationWillResignActive(const Sender: UIApplication); virtual; cdecl;
    procedure applicationWillTerminate(const Sender: UIApplication); virtual; cdecl;
  end;

  TApplicationDelegateClass = class of TApplicationDelegate;

  TObjectiveCLocal<T: IObjectiveCInstance> = class(TOCLocal)
  protected
    function GetObjectiveCClass: PTypeInfo; override;
  end;

  TBaseUIViewController<T: IObjectiveCInstance> = class(TObjectiveCLocal<T>)
  private
    function getView: UIView;
    procedure setView(const Value: UIView);
    function getViewController: UIViewController;
  public
    property view: UIView read getView write setView;
    property viewController: UIViewController read getViewController;
  end;

  TBaseUITableViewController<T: IObjectiveCInstance> = class(TBaseUIViewController<T>)
  private
    function getTableViewController: UITableViewController;
  public
    property tableViewController: UITableViewController
      read getTableViewController;
  end;

  TBaseUINavigationController<T: IObjectiveCInstance> = class(TBaseUIViewController<T>)
  private
    function getNavigationController: UINavigationController;
  public
    property navigationController: UINavigationController
      read getNavigationController;
  end;

var
  ApplicationDelegateClass: TApplicationDelegateClass = TApplicationDelegate;

function CocoaTouch: TCocoaTouch;

implementation

uses
  Macapi.ObjCRuntime,
  iOSapi.Utils;

const
  AppDelegateType = 'BlongAppDelegate';

type
{$M+}
  PUIApplication = Pointer;
  PNSDictionary = Pointer;

// Application delegates

function applicationDidFinishLaunchingWithOptions(self: id; _cmd: SEL;
  application: PUIApplication; options: PNSDictionary): Boolean; cdecl;
begin
  Result := CocoaTouch.FAppDelegate.application
    (TUIApplication.Wrap(application), TNSDictionary.Wrap(options));
end;

procedure applicationDidBecomeActive(self: id; _cmd: SEL;
  application: PUIApplication); cdecl;
begin
  CocoaTouch.FAppDelegate.applicationDidBecomeActive
    (TUIApplication.Wrap(application));
end;

procedure applicationDidEnterBackground(self: id; _cmd: SEL;
  application: PUIApplication); cdecl;
begin
  // This is called.
  CocoaTouch.FAppDelegate.applicationDidEnterBackground
    (TUIApplication.Wrap(application));
end;

procedure applicationWillEnterForeground(self: id; _cmd: SEL;
  application: PUIApplication); cdecl;
begin
  CocoaTouch.FAppDelegate.applicationWillEnterForeground
    (TUIApplication.Wrap(application));
end;

procedure applicationWillResignActive(self: id; _cmd: SEL;
  application: PUIApplication); cdecl;
begin
  // This seems not to be called.
  CocoaTouch.FAppDelegate.applicationWillResignActive
    (TUIApplication.Wrap(application));
end;

procedure applicationWillTerminate(self: id; _cmd: SEL;
  application: PUIApplication); cdecl;
begin
  CocoaTouch.FAppDelegate.applicationWillTerminate
    (TUIApplication.Wrap(application));
end;

procedure applicationDidReceiveMemoryWarning(self: id; _cmd: SEL;
  application: PUIApplication); cdecl;
begin
  CocoaTouch.FAppDelegate.applicationDidReceiveMemoryWarning
    (TUIApplication.Wrap(application));
end;

{ TCocoaTouch }

constructor TCocoaTouch.Create;
var
  appDelegateClass: Pointer;
begin
  inherited;

  FMainScreen := TUIScreen.Wrap(TUIScreen.OCClass.mainScreen);
  FCurrentDevice := TUIDevice.Wrap(TUIDevice.OCClass.currentDevice);

  FAppDelegate := ApplicationDelegateClass.Create;

  // Set up application delegate manually for now

  // Create a class to serve as our application delegate
  appDelegateClass := objc_allocateClassPair(objc_getClass('NSObject'), AppDelegateType, 0);

  // Add the UIApplicationDelegate protocol
  class_addProtocol(appDelegateClass, objc_getProtocol('UIApplicationDelegate'));

  // Add the "finish launching" delegate method
  class_addMethod(appDelegateClass, sel_getUid('application:didFinishLaunchingWithOptions:'),
    @applicationDidFinishLaunchingWithOptions, 'v@:@@');

  // Add additional application delegate methods
  class_addMethod(appDelegateClass, sel_getUid('applicationDidEnterBackground:'),
    @applicationDidEnterBackground, 'v@:@');
  class_addMethod(appDelegateClass, sel_getUid('applicationWillResignActive:'),
    @applicationWillResignActive, 'v@:@');
  class_addMethod(appDelegateClass, sel_getUid('applicationDidBecomeActive:'),
    @applicationDidBecomeActive, 'v@:@');
  class_addMethod(appDelegateClass, sel_getUid('applicationWillEnterForeground:'),
    @applicationWillEnterForeground, 'v@:@');
  class_addMethod(appDelegateClass, sel_getUid('applicationWillTerminate:'),
    @applicationWillTerminate, 'v@:@');
  class_addMethod(appDelegateClass, sel_getUid('applicationDidReceiveMemoryWarning:'),
    @applicationDidReceiveMemoryWarning, 'v@:@');

  // Register the delegate class
  objc_registerClassPair(appDelegateClass);
end;

var
  CocoaTouchObject: TCocoaTouch = nil;

function CocoaTouch: TCocoaTouch;
begin
  if CocoaTouchObject = nil then
    CocoaTouchObject := TCocoaTouch.Create;
  Result := CocoaTouchObject;
end;

procedure TCocoaTouch.Run;
begin
  //UIApplicationMain is documented to never return
  {ExitCode :=} UIApplicationMain(System.ArgCount, System.ArgValues, nil,
    PNSSTR(AppDelegateType));
end;

{ TApplicationDelegate }

constructor TApplicationDelegate.Create;
begin
  NSLog(PNSSTR(AppDelegateType + '.Create'));
end;

destructor TApplicationDelegate.Destroy;
begin
  //We don't get here
  NSLog(PNSSTR(AppDelegateType + '.Destroy'));
  inherited;
end;

function TApplicationDelegate.application(const Sender: UIApplication;
  const didFinishLaunchingWithOptions: NSDictionary): Boolean;
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationDidFinishLaunchingWithOptions'));
  CocoaTouchObject.FApplication := Sender;
  Result := true;
end;

procedure TApplicationDelegate.applicationDidBecomeActive(
  const Sender: UIApplication);
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationDidBecomeActive'));
  // Restart any tasks that were paused (or not yet started) while the
  // application was inactive. If the application was previously in the
  // background, optionally refresh the user interface.
end;

procedure TApplicationDelegate.applicationDidEnterBackground(
  const Sender: UIApplication);
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationDidEnterBackground'));
  // Use this method to release shared resources, save user data, invalidate
  // timers, and store enough application state information to restore your
  // application to its current state in case it is terminated later.
  // If your application supports background execution, this method is called
  // instead of applicationWillTerminate: when the user quits.
end;

procedure TApplicationDelegate.applicationDidReceiveMemoryWarning(
  const Sender: UIApplication);
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationDidReceiveMemoryWarning'));

end;

procedure TApplicationDelegate.applicationWillEnterForeground(
  const Sender: UIApplication);
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationWillEnterForeground'));
  // Called as part of the transition from the background to the inactive state;
  // here you can undo many of the changes made on entering the background.
end;

procedure TApplicationDelegate.applicationWillResignActive(
  const Sender: UIApplication);
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationWillResignActive'));
  // Sent when the application is about to move from active to inactive state.
  // This can occur for certain types of temporary interruptions (such as an
  // incoming phone call or SMS message) or when the user quits the application
  // and it begins the transition to the background state.
  // Use this method to pause ongoing tasks, disable timers, and throttle down
  // OpenGL ES frame rates. Games should use this method to pause the game.
end;

procedure TApplicationDelegate.applicationWillTerminate(
  const Sender: UIApplication);
begin
  NSLog(PNSSTR(AppDelegateType + '.applicationWillTerminate'));
  // Called when the application is about to terminate. Save data if appropriate
  // See also applicationDidEnterBackground:.
end;

{ TObjectiveCLocal<T> }

function TObjectiveCLocal<T>.GetObjectiveCClass: PTypeInfo;
begin
  //Return RTTI for the custom view controller interface
  Result := TypeInfo(T);
end;

{ TBaseUIViewController }

function TBaseUIViewController<T>.getView: UIView;
begin
  Result := viewController.view
end;

function TBaseUIViewController<T>.getViewController: UIViewController;
begin
  Result := Super as UIViewController
end;

procedure TBaseUIViewController<T>.setView(const Value: UIView);
begin
  viewController.setView(value)
end;

{ TBaseUITableViewController }

function TBaseUITableViewController<T>.getTableViewController: UITableViewController;
begin
  Result := Super as UITableViewController
end;

{ TBaseUINavigationController<T> }

function TBaseUINavigationController<T>.getNavigationController: UINavigationController;
begin
  Result := Super as UINavigationController
end;

initialization
finalization
  CocoaTouchObject := nil;
end.