unit FMX.MapView.iOS;

interface

procedure RegisterMapViewService;
procedure UnRegisterMapViewService;

implementation

uses
  System.Types, System.Classes, System.Sensors,
  Macapi.ObjectiveC,
  iOSapi.Foundation, iOSapi.UIKit, iOSapi.MapKit, iOSapi.CoreLocation,
  iOSapi.Utils,
  FMX.Types, FMX.Platform, FMX.MapView, FMX.Forms, FMX.Platform.iOS,
  FMX.Utils;

type
  TiOSMapViewService = class;

{ TiOSMapViewDelegate }

  TiOSMapViewDelegate = class(TOCLocal, MKMapViewDelegate)
  private
    //NOTE: FMX WebBrowser equiv not marked weak
    //Without this the TiOSMapViewService will not get destroyed through mutual referencing
    [weak]FMapView: TiOSMapViewService;
  public
    destructor Destroy; override;
    procedure SetMapView(const AMapView: TiOSMapViewService);
    //TODO: Add declarations for the many other delegate methods in MKMapViewDelegate
    procedure mapViewWillStartLoadingMap(mapView: MKMapView); cdecl;
    procedure mapViewDidFinishLoadingMap(mapView: MKMapView); cdecl;
    procedure mapViewDidFailLoadingMap(mapView: MKMapView; withError: NSError); cdecl;
  end;

  TiOSMVService = class(TMVFactoryService)
  protected
    function DoCreateMapView: ICustomMapView; override;
  end;

{ TiOSMapViewService }

  TiOSMapViewService = class(TInterfacedObject, ICustomMapView)
  private
    FMapView: MKMapView;
    FMapControl: TCustomMapView;
    FDelegate: TiOSMapViewDelegate;
  protected
    procedure Show;
    procedure Hide;

    //ICustomMapView methods
    procedure SetMapViewControl(const AValue: TCustomMapView);
    function GetParent : TFmxObject;
    function GetVisible : Boolean;
    procedure UpdateContentFromControl;
    function GetMapType: TMapType;
    procedure SetMapType(Value: TMapType);

    procedure StartLoading;
    procedure FinishLoading;
    procedure FailLoadingWithError(const ErrorMsg: string);
  public
    constructor Create;
    destructor Destroy; override;

    property MapType: TMapType read GetMapType write SetMapType;
    function ShowsUserLocation: Boolean;
    procedure SetShowsUserLocation(Value: Boolean);
    procedure SetCenterCoordinate(coordinate: TLocationCoord2D; animated: Boolean);
    procedure SetRegion(region: TCoord2DRegion; animated: Boolean);
    procedure SetUserLocationAnnotation;
    procedure SetUserLocationText(Title, Subtitle: String);
  end;

var
  MVService : TiOSMVService;

procedure RegisterMapViewService;
begin
  MVService := TiOSMVService.Create;
  TPlatformServices.Current.AddPlatformService(IFMXMVService, MVService);
end;

procedure UnRegisterMapViewService;
begin
  TPlatformServices.Current.RemovePlatformService(IFMXMVService);
end;

{ TiOSMapViewDelegate }

destructor TiOSMapViewDelegate.Destroy;
begin
  FMapView := nil;
  inherited;
end;

procedure TiOSMapViewDelegate.SetMapView(const AMapView: TiOSMapViewService);
begin
  FMapView := AMapView
end;

procedure TiOSMapViewDelegate.mapViewWillStartLoadingMap(mapView: MKMapView);
begin
  if Assigned(FMapView) then
    FMapView.StartLoading;
end;

procedure TiOSMapViewDelegate.mapViewDidFinishLoadingMap(mapView: MKMapView);
begin
  if Assigned(FMapView) then
    FMapView.FinishLoading;
end;

procedure TiOSMapViewDelegate.mapViewDidFailLoadingMap(mapView: MKMapView;
  withError: NSError);
begin
  if Assigned(FMapView) then
    FMapView.FailLoadingWithError(NSStrToStr(withError.localizedDescription));
end;

{ TiOSMVService }

function TiOSMVService.DoCreateMapView: ICustomMapView;
begin
  Result := TiOSMapViewService.Create;
end;

{ TiOSMapViewService }

constructor TiOSMapViewService.Create;
begin
  inherited;
  FMapView := TMKMapView.Create;
  FDelegate := TiOSMapViewDelegate.Create;
  FDelegate.SetMapView(Self);
  FMapView.setDelegate(FDelegate.GetObjectID);
end;

destructor TiOSMapViewService.Destroy;
begin
  FMapView.setDelegate(nil);
  FDelegate.Free;

  if Assigned(FMapView) then
  begin
    //Let's lose the view hierarchy reference to the MKMapView
    FMapView.removeFromSuperview;
    //Now release our reference to the MKMapView
    FMapView := nil;
  end;

  inherited;
end;

procedure TiOSMapViewService.FailLoadingWithError(const ErrorMsg: string);
begin
  if Assigned(FMapControl) then
    FMapControl.FailLoadingWithError(ErrorMsg);
end;

procedure TiOSMapViewService.FinishLoading;
begin
  if Assigned(FMapControl) then
    FMapControl.FinishLoading;
end;

function TiOSMapViewService.GetMapType: TMapType;
begin
  Result := mtHybrid;
  if Assigned(FMapView) then
    case FMapView.mapType of
      MKMapTypeStandard: Result := mtStandard;
      MKMapTypeSatellite: Result := mtSatellite;
    else
      {MKMapTypeHybrid:} Result := mtHybrid;
    end;
end;

function TiOSMapViewService.GetParent: TFmxObject;
begin
  Result := FMapControl.Parent;
end;

function TiOSMapViewService.GetVisible: Boolean;
begin
  Result := False;
  if Assigned(FMapControl) then
    Result := FMapControl.Visible;
end;

procedure TiOSMapViewService.Hide;
begin
  if Assigned(FMapView) and (not FMapView.isHidden) then
    FMapView.setHidden(True);
end;

procedure TiOSMapViewService.SetCenterCoordinate(
  coordinate: TLocationCoord2D; animated: Boolean);
begin
  if Assigned(FMapView) then
    FMapView.setCenterCoordinate(ConvCoord2CLLocationCoord(coordinate), animated);
end;

procedure TiOSMapViewService.SetMapType(Value: TMapType);
begin
  if Assigned(FMapView) then
    FMapView.SetMapType(Integer(Value))
end;

procedure TiOSMapViewService.SetMapViewControl(const AValue: TCustomMapView);
begin
  FMapControl := AValue;
end;

procedure TiOSMapViewService.SetRegion(region: TCoord2DRegion;
  animated: Boolean);
begin
  if Assigned(FMapView) then
    FMapView.setRegion(ConvCoordRegion2MKCoordRegion(region), animated);
end;

procedure TiOSMapViewService.SetShowsUserLocation(Value: Boolean);
begin
  if Assigned(FMapView) then
    FMapView.setShowsUserLocation(Value);
end;

procedure TiOSMapViewService.SetUserLocationAnnotation;
begin
  if Assigned(FMapView) then
    FMapView.selectAnnotation(FMapView.userLocation, True);
end;

procedure TiOSMapViewService.SetUserLocationText(Title, Subtitle: String);
begin
  if Assigned(FMapView) then
  begin
    FMapView.userLocation.setTitle(NSSTR(Title));
    FMapView.userLocation.setSubtitle(NSSTR(Subtitle));
  end;
end;

procedure TiOSMapViewService.Show;
begin
  if Assigned(FMapView) and (FMapView.isHidden) then
    FMapView.setHidden(False);
end;

function TiOSMapViewService.ShowsUserLocation: Boolean;
begin
  Result := False;
  if Assigned(FMapView) then
    Result := FMapView.showsUserLocation;
end;

procedure TiOSMapViewService.StartLoading;
begin
  if Assigned(FMapControl) then
    FMapControl.StartLoading;
end;

procedure TiOSMapViewService.UpdateContentFromControl;
var
  View: UIView;
  Bounds: TRectF;
  Form: TCommonCustomForm;
begin
  if Assigned(FMapView) then
  begin
    if (FMapControl <> nil)
      and not (csDesigning in FMapControl.ComponentState)
      and (FMapControl.Root <> nil)
      and (FMapControl.Root.GetObject is TCommonCustomForm) then
    begin
      Form := TCommonCustomForm(FMapControl.Root.GetObject);
      Bounds := TRectF.Create(
        FMapControl.Position.X, FMapControl.Position.Y,
        FMapControl.Width, FMapControl.Height);
      Bounds.Fit(FMapControl.AbsoluteRect);
      View := WindowHandleToPlatform(Form.Handle).View;
      View.addSubview(FMapView);
      FMapView.setFrame(CGRectFromRect(Bounds));
      FMapView.setHidden(not FMapControl.ParentedVisible);
    end
    else
      FMapView.setHidden(True);
  end;
end;

end.
