unit FMX.MapView;

interface

uses
  System.Classes, System.Generics.Collections, System.Sensors,
  FMX.Controls, FMX.Types, FMX.Utils;

type
  TCustomMapView = class;

  TMapViewDidFailLoadingMapWithErrorEvent = procedure(ASender: TObject; const ErrorMsg: string) of object;

  TMapType = (mtStandard, mtSatellite, mtHybrid);

  ICustomMapView = interface(IInterface)
  ['{86C3F82A-87D0-484C-9C2E-239C4D92CC85}']
    procedure SetMapViewControl(const AValue: TCustomMapView);
    function GetParent : TFmxObject;
    function GetVisible : Boolean;
    procedure UpdateContentFromControl;
    //iOS members surfaced to FMX
    function GetMapType: TMapType;
    procedure SetMapType(Value: TMapType);
    property MapType: TMapType read GetMapType write SetMapType;
    function ShowsUserLocation: Boolean;
    procedure SetShowsUserLocation(Value: Boolean);
    procedure SetCenterCoordinate(coordinate: TLocationCoord2D; animated: Boolean);
    procedure SetRegion(region: TCoord2DRegion; animated: Boolean);
    procedure SetUserLocationAnnotation;
    procedure SetUserLocationText(Title, Subtitle: String);
  end;

  IFMXMVService = interface (IInterface)
    ['{DD93AD5E-BB7F-471B-8B68-D7D1CD460CAB}']
    function CreateMapView: ICustomMapView;
    {
    procedure ReAlignMapViews;
    }
  end;

  TCustomMapView = class(TControl)
  private
    FMap: ICustomMapView;
    FOnWillStartLoadingMap:        TNotifyEvent;
    FOnDidFinishLoadingMap:        TNotifyEvent;
    FOnDidFailLoadingMapWithError: TMapViewDidFailLoadingMapWithErrorEvent;
    procedure UpdateContent;
    function GetMapType: TMapType;
    procedure SetMapType(const Value: TMapType);
    function GetShowsUserLocation: Boolean;
    procedure SetShowsUserLocation(Value: Boolean);
  protected
    procedure Move; override;
    procedure Resize; override;
    procedure Paint; override;
    procedure Show; override;
    procedure Hide; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;

    // getters and setters for web browser events

    function GetOnWillStartLoadingMap: TNotifyEvent;
    procedure SetOnWillStartLoadingMap(const AEvent: TNotifyEvent);
    function GetOnDidFinishLoadingMap: TNotifyEvent;
    procedure SetOnDidFinishLoadingMap(const AEvent: TNotifyEvent);
    function GetOnDidFailLoadingMapWithError: TMapViewDidFailLoadingMapWithErrorEvent;
    procedure SetOnDidFailLoadingMapWithError(const AEvent: TMapViewDidFailLoadingMapWithErrorEvent);
    procedure SetVisible(const Value: Boolean); override;

    property MapType: TMapType read GetMapType write SetMapType;
    property ShowsUserLocation: Boolean read GetShowsUserLocation write SetShowsUserLocation;

    property OnWillStartLoadingMap: TNotifyEvent read GetOnWillStartLoadingMap write SetOnWillStartLoadingMap;
    property OnDidFinishLoadingMap: TNotifyEvent read GetOnDidFinishLoadingMap write SetOnDidFinishLoadingMap;
    property OnDidFailLoadingMapWithError: TMapViewDidFailLoadingMapWithErrorEvent read GetOnDidFailLoadingMapWithError write SetOnDidFailLoadingMapWithError;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    //TODO
    procedure SetCenterCoordinate(coordinate: TLocationCoord2D; animated: Boolean);
    procedure SetRegion(region: TCoord2DRegion; animated: Boolean);
    procedure SetUserLocationAnnotation;
    procedure SetUserLocationText(Title, Subtitle: String);

    procedure StartLoading;
    procedure FinishLoading;
    procedure FailLoadingWithError(const ErrorMsg: string);
  end;

{ TMapView }
  [ComponentPlatformsAttribute(pidiOSSimulator or pidiOSDevice)]
  TMapView = class(TCustomMapView)
  published
    property Align;
    property Anchors;
    property Width;
    property Height;
    property Margins;
    property Position;
    property Visible default True;
    property MapType;
    property ShowsUserLocation;
    property OnWillStartLoadingMap;
    property OnDidFinishLoadingMap;
    property OnDidFailLoadingMapWithError;
  end;

{ TWBFactoryService }

  TMVFactoryService = class abstract (TInterfacedObject, IFMXMVService)
  protected
    FMapViews: TList<ICustomMapView>;
    function DoCreateMapView: ICustomMapView; virtual; abstract;
    procedure DoRemoveMapView(const AMapView: ICustomMapView);
    {
    procedure ReAlignMapViews;
    }
  public
    constructor Create;
    destructor Destroy; override;
    function CreateMapView: ICustomMapView;
  end;

implementation

uses
  System.Types, System.SysUtils,
{$IFDEF IOS}
  FMX.MapView.iOS,
{$ENDIF IOS}
  FMX.Graphics,
  FMX.Platform;

{ TCustomMapView }

constructor TCustomMapView.Create(AOwner: TComponent);
var
  MVService : IFMXMVService;
begin
  inherited;
  if TPlatformServices.Current.SupportsPlatformService(IFMXMVService, IInterface(MVService)) then
  begin
    FMap := MVService.CreateMapView;
    FMap.SetMapViewControl(Self);
  end;
end;

destructor TCustomMapView.Destroy;
begin
  FMap := nil;
  inherited;
end;

procedure TCustomMapView.FailLoadingWithError(const ErrorMsg: string);
begin
  if Assigned(FOnDidFailLoadingMapWithError) then
    FOnDidFailLoadingMapWithError(Self, ErrorMsg);
end;

procedure TCustomMapView.FinishLoading;
begin
  if Assigned(FOnDidFinishLoadingMap) then
    FOnDidFinishLoadingMap(Self)
end;

function TCustomMapView.GetMapType: TMapType;
begin
  Result := FMap.MapType
end;

function TCustomMapView.GetOnDidFailLoadingMapWithError
  : TMapViewDidFailLoadingMapWithErrorEvent;
begin
  Result := FOnDidFailLoadingMapWithError
end;

function TCustomMapView.GetOnDidFinishLoadingMap: TNotifyEvent;
begin
  Result := FOnDidFinishLoadingMap
end;

function TCustomMapView.GetOnWillStartLoadingMap: TNotifyEvent;
begin
  Result := FOnWillStartLoadingMap
end;

procedure TCustomMapView.Hide;
begin
  inherited;
  UpdateContent;
end;

procedure TCustomMapView.Move;
begin
  inherited;
  UpdateContent;
end;

procedure TCustomMapView.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;

end;

procedure TCustomMapView.Paint;
var
  R: TRectF;
begin
  if (csDesigning in ComponentState) and not Locked and not FInPaintTo then
  begin
    R := LocalRect;
    InflateRect(R, -0.5, -0.5);
    Canvas.Stroke.Thickness := 1;
    Canvas.Stroke.Dash := TStrokeDash.sdDash;
    Canvas.Stroke.Kind := TBrushKind.bkSolid;
    Canvas.Stroke.Color := $A0909090;
    Canvas.DrawRect(R, 0, 0, AllCorners, AbsoluteOpacity);
    Canvas.Stroke.Dash := TStrokeDash.sdSolid;
  end;
end;

procedure TCustomMapView.Resize;
begin
  inherited;
  UpdateContent;
end;

procedure TCustomMapView.SetCenterCoordinate(coordinate: TLocationCoord2D;
  animated: Boolean);
begin
  if Assigned(FMap) then
    FMap.SetCenterCoordinate(coordinate, animated);
end;

procedure TCustomMapView.SetMapType(const Value: TMapType);
begin
  if Assigned(FMap) then
    FMap.MapType := Value
end;

procedure TCustomMapView.SetOnDidFailLoadingMapWithError(
  const AEvent: TMapViewDidFailLoadingMapWithErrorEvent);
begin
  FOnDidFailLoadingMapWithError := AEvent
end;

procedure TCustomMapView.SetOnDidFinishLoadingMap(
  const AEvent: TNotifyEvent);
begin
  FOnDidFinishLoadingMap := AEvent
end;

procedure TCustomMapView.SetOnWillStartLoadingMap(
  const AEvent: TNotifyEvent);
begin
  FOnWillStartLoadingMap := AEvent
end;

procedure TCustomMapView.SetRegion(region: TCoord2DRegion;
  animated: Boolean);
begin
  if Assigned(FMap) then
    FMap.SetRegion(region, animated);
end;

procedure TCustomMapView.SetShowsUserLocation(Value: Boolean);
begin
  if Assigned(FMap) then
    FMap.SetShowsUserLocation(Value);
end;

procedure TCustomMapView.SetUserLocationAnnotation;
begin
  if Assigned(FMap) then
    FMap.SetUserLocationAnnotation;
end;

procedure TCustomMapView.SetUserLocationText(Title, Subtitle: String);
begin
  if Assigned(FMap) then
    FMap.SetUserLocationText(Title, Subtitle);
end;

procedure TCustomMapView.SetVisible(const Value: Boolean);
begin
  inherited;

end;

procedure TCustomMapView.Show;
begin
  inherited;
  UpdateContent;
end;

function TCustomMapView.GetShowsUserLocation: Boolean;
begin
  Result := False;
  if Assigned(FMap) then
    Result := FMap.ShowsUserLocation;
end;

procedure TCustomMapView.StartLoading;
begin
  if Assigned(FOnWillStartLoadingMap) then
    FOnWillStartLoadingMap(Self)
end;

procedure TCustomMapView.UpdateContent;
begin
  if Assigned(FMap) then
    FMap.UpdateContentFromControl;
end;

{ TMVFactoryService }

constructor TMVFactoryService.Create;
begin
  inherited Create;
  FMapViews := TList<ICustomMapView>.Create;
end;

destructor TMVFactoryService.Destroy;
begin
  FreeAndNil(FMapViews);
  inherited Destroy;
end;

function TMVFactoryService.CreateMapView: ICustomMapView;
begin
  Result := DoCreateMapView;
  //NOTE: This is what happens with web browsers - they are added into a list
  //that is not accessible and not used. This keeps the ref count up on each
  //one and causes issues as they do not get destroyed. Let's skip that part
  //and that problem
  //FMapViews.Add(Result);
end;

procedure TMVFactoryService.DoRemoveMapView(const AMapView: ICustomMapView);
begin
  if Assigned(FMapViews) and Assigned(AMapView) then
    FMapViews.Remove(AMapView);
end;

{
procedure TMVFactoryService.ReAlignMapViews;
var
  MapView : ICustomMapView;
  Parent : TFmxObject;
begin
  for MapView in FWebBrowsers do
  begin
    Parent := MapView.GetParent;
    if Assigned(Parent) and (Parent is TControl) then
    begin
      if TControl(Parent).ParentedVisible then
      begin
        if MapView.GetVisible then
          MapView.Show
        else
          MapView.Hide;
      end
      else
      begin
        MapView.Hide;
      end;
    end
    else
      MapView.Hide;
    if MapView.GetVisible then
      MapView.UpdateContentFromControl;
  end;
end;
}

initialization

{$IFDEF IOS}
  RegisterMapViewService;
{$ENDIF IOS}

  RegisterFmxClasses([TMapView]);

end.
