unit iOSapi.Utils;

interface

uses
  System.Types,
  System.Sensors,
  System.RTTI,
  MacApi.ObjectiveC,
  Macapi.ObjCRuntime,
{$IFDEF USING_FMX}
  FMX.Utils,
  FMX.Types,
{$ENDIF}
  iOSAPI.Foundation,
  iOSAPI.UIKit,
  iOSapi.CoreGraphics,
  iOSapi.CoreLocation,
  iOSapi.MapKit;

procedure Alert(Caption, Msg: String);

function CGRectFromRect(const R: TRectF): CGRect;

{$IFDEF USING_FMX}
function ConvCLLocationCoord2Coord(const Value: CLLocationCoordinate2D): TLocationCoord2D;
function ConvCoord2CLLocationCoord(const Value: TLocationCoord2D): CLLocationCoordinate2D;
function ConvCoordSpan2MKCoordSpan(const Value: TCoord2DSpan): MKCoordinateSpan;
function ConvCoordRegion2MKCoordRegion(const Value: TCoord2DRegion): MKCoordinateRegion;
{$ENDIF}

{$IF CompilerVersion = 25}
///	<summary>
///	  Return global instance of UIApplication. The equivalent of this in
///	  Objective-C:<br /><br />[UIApplication sharedApplication];
///	</summary>
///	<remarks>
///	  A shared/global UIApplication instance
///	</remarks>
function SharedApplication: UIApplication;
{$ENDIF}

///	<summary>
///	  Take a Delphi string nd turn it into something usable as an iOS NSString
///	  in an imported iOS API - that would be a PNSString (actually just a
///	  pointer in Delphi terms)
///	</summary>
///	<param name="AStr">
///	  The input string
///	</param>
///	<returns>
///	  A PNSString pointer that can be fed into an iOS API that requires an
///	  NSString in iOS, or a PNSString in Delphi
///	</returns>
function PNSStr(const AStr: String): PNSString;

///	<summary>
///	  Gets a Delphi string from an Objective-C NSString
///	</summary>
///	<param name="ASource">
///	  An Objective-C NSString
///	</param>
///	<returns>
///	  A Delphi string with the same content as the input NSString
///	</returns>
function NSStrToStr(const ASource: NSString): string;

function SelectorForMethod(const Obj: TObject; const MethodName: String): SEL; overload;
function SelectorForMethod(const Cls: TClass; const MethodName: String): SEL; overload;
function SelectorForMethod(const Method: TRttiMethod): SEL; overload;

{$IFDEF USING_FMX}
procedure TidyAwayIOSView(Handle: TWindowHandle);
{$ENDIF}

implementation

uses
{$IFDEF USING_FMX}
  FMX.Platform.iOS,
{$ENDIF}
  System.SysUtils;

procedure Alert(Caption, Msg: String);
var
  alert: UIAlertView;
begin
  alert := TUIAlertView.Wrap(TUIAlertView.Alloc.initWithTitle(
             {title}NSSTR(caption),
             {message}NSSTR(msg),
             {delegate}nil,
             {cancelButtonTitle}NSSTR('OK'),
             {otherButtonTitles}nil));
  alert.show;
  alert.release;
  alert := nil;
end;

function CGRectFromRect(const R: TRectF): CGRect;
begin
  Result.origin.x := R.Left;
  Result.origin.Y := R.Top;
  Result.size.Width := R.Right - R.Left;
  Result.size.Height := R.Bottom - R.Top;
end;

{$IFDEF USING_FMX}
function ConvCLLocationCoord2Coord(const Value: CLLocationCoordinate2D): TLocationCoord2D;
begin
  Result := TLocationCoord2D.Create(Value.latitude, Value.longitude);
end;

function ConvCoord2CLLocationCoord(const Value: TLocationCoord2D): CLLocationCoordinate2D;
begin
  Result.latitude := Value.Latitude;
  Result.longitude := Value.Longitude
end;

function ConvCoordSpan2MKCoordSpan(const Value: TCoord2DSpan): MKCoordinateSpan;
begin
  Result.latitudeDelta := Value.LatitudeDelta;
  Result.longitudeDelta := Value.LongitudeDelta
end;

function ConvCoordRegion2MKCoordRegion(const Value: TCoord2DRegion): MKCoordinateRegion;
begin
  Result.center := ConvCoord2CLLocationCoord(Value.Center);
  Result.span := ConvCoordSpan2MKCoordSpan(Value.Span);
end;
{$ENDIF}

{$IF CompilerVersion = 25)}
//Equivalent function was added to FMX.helpers.iOS in XE5 (v26)
function SharedApplication: UIApplication;
begin
  Result := TUIApplication.Wrap(TUIApplication.OCClass.sharedApplication);
end;
{$ENDIF}

function PNSStr(const AStr: String): PNSString;
begin
  Result := (NSSTR(AStr) as ILocalObject).GetObjectID
end;

function NSStrToStr(const ASource: NSString): string;
begin
  Result := string(ASource.UTF8String);
end;

function GetMethodRTTI(const Cls: TClass; const MethodName: String): TRttiMethod;
var
  ctx: TRttiContext;
  typ: TRttiType;
begin
  Result := nil;
  ctx := TRttiContext.Create;
  typ := ctx.GetType(Cls);
  if Assigned(typ) then
    Result := typ.GetMethod(MethodName);
end;

function SelectorForMethod(const Obj: TObject; const MethodName: String): SEL;
begin
  Result := SelectorForMethod(GetMethodRTTI(Obj.ClassType, MethodName))
end;

function SelectorForMethod(const Cls: TClass; const MethodName: String): SEL;
begin
  Result := SelectorForMethod(GetMethodRTTI(Cls, MethodName))
end;

function SelectorForMethod(const Method: TRttiMethod): SEL;
var
  MangledName: string;
  M: TMarshaller;
begin
  MangledName := MangleMethodName(Method);
  Result := sel_registerName(M.AsAnsi(MangledName, CP_UTF8).ToPointer);
end;

{$IFDEF USING_FMX}
procedure TidyAwayIOSView(Handle: TWindowHandle);
var
  View: UIView;
begin
  View := WindowHandleToPlatform(Handle).View;
  View.removeFromSuperview;
end;
{$ENDIF}

end.
