unit MainCommands;

// This is an extension/evolution of Jim McKeeth's VoiceCommands demo found online at:
// https://github.com/jimmckeeth/FireMonkey-Android-Voice/tree/master/Demos/VoiceCommands

interface

uses
  System.SysUtils, System.Types, System.UITypes, System.Classes, System.Variants,
  FMX.Types, FMX.Controls, FMX.Forms, FMX.Graphics, FMX.Dialogs, FMX.Layouts,
  FMX.ListBox, SpeechRecognition, AndroidTTS, FMX.StdCtrls,
  FMX.Controls.Presentation;

{$SCOPEDENUMS ON}
type
  TMenuState = (Top, Colour, Reset, FullScreen);

  TMainForm = class(TForm)
    ListBox1: TListBox;
    Layout1: TLayout;
    ListenButton: TButton;
    procedure ListenButtonClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private declarations }
    SpeechRecognition: TSpeechRecognition;
    AndroidTTS: TAndroidTTS;
    FMenuState: TMenuState;
    FMenuItems: array [TMenuState] of TGuesses;
    procedure SetMenu(const items: TGuesses);
    procedure SpeechRecognition1RecognitionEx(Sender: TObject;
      Guesses: TRecognitionResults);
    procedure SpeechRecognition1Command(Sender: TObject; Command: string);
  public
    { Public declarations }
  end;

var
  MainForm: TMainForm;

implementation

uses
  LaunchActivities;

{$R *.fmx}

const
  cmdReset = 'Reset';
  cmdColour = 'Colour';
  cmdFullScreen = 'Full screen';
  cmdHowAreYou = 'How are you';
  cmdOpenThePodBayDoors = 'Open the pod bay doors Hal';
  cmdWhatsTheproblem = 'What''s the problem';
  // Actually "I don't know what you're talking about, Hal." but the SR will recognose this version
  cmdIDontKnowWhatYoureTalkingAbout = 'I don''t know what you''re talking about how';
  // Actually "I won't argue with you anymore. Open the doors." but the SR will recognose this version
  cmdIWontArgueWithYouAnyMore = 'I want to argue with you anymore Open the doors';
  cmdMirrorMirror = 'Mirror mirror on the wall who''s the fairest of them all';
  cmdTime = 'What''s the time';

procedure TMainForm.FormCreate(Sender: TObject);
begin
  SpeechRecognition := TSpeechRecognition.Create(Self);
  SpeechRecognition.Prompt := 'Your command?';
  SpeechRecognition.Language := 'en-GB';
  SpeechRecognition.OnRecognitionEx := SpeechRecognition1RecognitionEx;
  SpeechRecognition.OnCommand := SpeechRecognition1Command;
  FMenuItems[TMenuState.Top] := [
    cmdReset,
    cmdColour,
    cmdFullScreen,
    cmdHowAreYou,
    cmdOpenThePodBayDoors,
    cmdWhatsTheproblem,
    cmdIDontKnowWhatYoureTalkingAbout,
    // Actually "I won't argue with you anymore. Open the doors." but the SR will recognose this version
    cmdIWontArgueWithYouAnyMore,
    cmdMirrorMirror,
    cmdTime];
  FMenuItems[TMenuState.Colour]     := ['Red', 'Blue', 'Green', 'Yellow'];
  FMenuItems[TMenuState.FullScreen] := ['Full', 'Restore'];
  FMenuItems[TMenuState.Reset]      := ['Yes', 'No'];

  AndroidTTS := TAndroidTTS.Create(Self);
end;

procedure TMainForm.ListenButtonClick(Sender: TObject);
begin
  ListBox1.Clear;
  SetMenu(FMenuItems[FMenuState]);
  Application.ProcessMessages;
  SpeechRecognition.ListenFor(FMenuItems[FMenuState]);
end;

{$REGION 'Time code'}
function TimeToNaturalStr(Time: TDateTime; AddDayPartSuffix: Boolean = True): String;
const
  Near: array[0..4] of string = (
    '', ' just gone', ' just after', ' nearly', ' almost');

  Hours: array[0..11] of string = ('twelve', 'one', 'two', 'three',
    'four', 'five', 'six', 'seven', 'eight', 'nine', 'ten', 'eleven');

  Mins: array[0..11] of String = (' ', ' five past ', ' ten past ',
    ' a quarter past ', ' twenty past ', ' twenty five past ', ' half past ',
    ' twenty five to ', ' twenty to ', ' a quarter to ', ' ten to ',
    ' five to ');

  TwelveHours: array[Boolean] of String = ('midday', 'midnight');
var
  Hour, Min, Sec, MSec: Word;
begin
  DecodeTime(Time, Hour, Min, Sec, MSec);
  //Round to nearest minute
  if MSec > 500 then Inc(Sec);
  if Sec > 29 then Inc(Min);
  //Cater for being TO the next hour
  if Min > 32 then
    Inc(Hour);
  //Relate minute to an analog reference to 5 min blocks
  Result := 'It''s' + Near[Min mod 5];
  //Move minutes forward, so it's easier to find the closest 5 minute point
  Inc(Min, 2);
  //Find the nearest five minutes
  Dec(Min, Min mod 5);
  Result := Result + Mins[(Min div 5) mod 12];
  //Replace 12 o'clock with midday/midnight
  if Hour in [0, 12] then
    Hours[0] := TwelveHours[Hour = 0];
  Result := Result + Hours[Hour mod 12];
  if (Min = 0) and not (Hour in [0, 12]) then
    Result := Result + ' o''clock';
  //Only add morning/afternoon/evening/night if not saying midday/night
  if AddDayPartSuffix and ((Hour mod 12) > 0) then
    if (Hour < 12) or ((Hour = 12) and (Min > 32)) then
      Result := Result + ' in the morning'
    else
      if ((Hour = 12) and (Min <= 32)) or (Hour in [13..17]) then
        Result := Result + ' in the afternoon'
      else
        if (Hour in [18..20]) then
          Result := Result + ' in the evening'
        else
          Result := Result + ' at night'
end;
{$ENDREGION}

procedure TMainForm.SetMenu(const items: TGuesses);
var
  i: Integer;
begin
  ListBox1.Items.Clear;
  for i := 0 to length(items) - 1 do
    ListBox1.Items.Add(items[i]);
end;

procedure TMainForm.SpeechRecognition1Command(Sender: TObject; Command: string);
begin
  if Command = cmdReset then FMenuState := TMenuState.Reset else
  if Command = cmdColour then FMenuState := TMenuState.Colour else
  if Command = cmdFullScreen then FMenuState := TMenuState.FullScreen else
  if Command = cmdOpenThePodBayDoors then
  begin
    //LaunchURL('http://www.moviesoundscentral.com/sounds/2001/hal.wav');
    PlayWAV('raw/hal');
  end
  else
  if Command = cmdWhatsTheproblem then
  begin
    PlayWAV('raw/knowproblem');
  end
  else
  if (Command = cmdIWontArgueWithYouAnyMore) or
     (Command = cmdIDontKnowWhatYoureTalkingAbout) then
  begin
    PlayWAV('raw/bye');
  end
  else
  if Command = cmdHowAreYou then
  begin
    case Random(5) of
      0: PlayWAV('raw/better');
      1: PlayWAV('raw/fault');
      2: PlayWAV('raw/function');
      3: PlayWAV('raw/operatnl');
      4: PlayWAV('raw/info');
    end;
  end
  else
  if Command = cmdMirrorMirror then
  begin
    AndroidTTS.Speak('Why, Brian, the answer is *you*, of course! You are the most devilishly handsome brute!');
  end
  else
  if Command = cmdTime then
  begin
    AndroidTTS.Speak(TimeToNaturalStr(Time));
  end
  else
  if Command = 'Full' then
  begin
    FullScreen := True;
    FMenuState := TMenuState.Top;
  end else
  if Command = 'Restore' then
  begin
    FullScreen := False;
    FMenuState := TMenuState.Top;
  end else
  if Command = 'Yes' then
  begin
    FullScreen := False;
    ListenButton.TintColor := TAlphaColorRec.Null;
    FMenuState := TMenuState.Top;
  end else
  if Command = 'Red' then
  begin
    FMenuState := TMenuState.Top;
    ListenButton.TintColor := TAlphaColorRec.Red;
  end else
  if Command = 'Yellow' then
  begin
    FMenuState := TMenuState.Top;
    ListenButton.TintColor := TAlphaColorRec.Yellow;
  end else
  if Command = 'Blue' then
  begin
    FMenuState := TMenuState.Top;
    ListenButton.TintColor := TAlphaColorRec.Blue;
  end else
  if Command = 'Green' then
  begin
    FMenuState := TMenuState.Top;
    ListenButton.TintColor := TAlphaColorRec.Green;
  end
  else
    ListBox1.Items.Add('Not found!');
  ListBox1.Items.Add('>' + Command + '<');
end;

procedure TMainForm.SpeechRecognition1RecognitionEx(Sender: TObject;
  Guesses: TRecognitionResults);
var
  I, Index: Integer;
  Item: TListBoxItem;
begin
  for I := 0 to Pred(Length(Guesses)) do
  begin
    Index := ListBox1.Items.Add('');
    Item := ListBox1.ListItems[Index];
    Item.ItemData.Text := Guesses[I].Guess;
    Item.ItemData.Detail := Format('%.1f%%', [Guesses[I].Confidence * 100]);
  end;
end;

end.