unit AndroidTTS;

// This is a version of Jim McKeeth's Text To Speech component, which can be found online at:
// https://github.com/jimmckeeth/FireMonkey-Android-Voice/tree/master/Components

interface

uses
  System.SysUtils, System.Classes,
  {$IFDEF ANDROID}
  AndroidAPI.JNIBridge, Androidapi.JNI.Speech,
  Androidapi.JNI.JavaTypes,
  {$ENDIF}
  FMX.Forms;

type
  TAndroidTTS = class(TComponent)
  private
  {$IFDEF ANDROID}
    FTTS: JTextToSpeech;
    type
      TTTSOnInitListener = class(TJavaLocal, JTextToSpeech_OnInitListener)
      private
        [weak] FParent : TAndroidTTS;
      public
        constructor Create(AParent: TAndroidTTS);
        // JTextToSpeech_OnInitListener
        procedure onInit(status: Integer); cdecl;
      end;
      TTTSOnUtteranceCompletedListener = class(TJavaLocal,
        JTextToSpeech_OnUtteranceCompletedListener)
      private
        [weak] FParent : TAndroidTTS;
      public
        constructor Create(AParent: TAndroidTTS);
        // JTextToSpeech_OnUtteranceCompletedListener
        procedure onUtteranceCompleted(utteranceID: JString); cdecl;
      end;

    private
      FTTSOnInitListener : TTTSOnInitListener;
      FTTSOnUtteranceCompletedListener : TTTSOnUtteranceCompletedListener;
      FDone: TNotifyEvent;
    public
      property OnDone: TNotifyEvent read FDone write FDone;
{$ENDIF}
      constructor Create(AOwner: TComponent); override;
      procedure Speak(say: String);
    end;

procedure Register;

implementation

uses
  FMX.Types,
{$IFDEF ANDROID}
  FMX.Helpers.Android
, FMX.Platform.Android
, Androidapi.JNI.Os
, Androidapi.JNI.GraphicsContentViewText
, Androidapi.Helpers
, Androidapi.JNI.App
;

  {$ENDIF}

procedure Register;
begin
  RegisterComponents('Android', [TAndroidTTS]);
end;

{ TAndroidTTS }

constructor TAndroidTTS.Create(AOwner: TComponent);
begin
  inherited;
{$IFDEF ANDROID}
  FTTSOnInitListener := TTTSOnInitListener.Create(Self);
  FTTSOnUtteranceCompletedListener := TTTSOnUtteranceCompletedListener.Create(Self);
  FTTS := TJTextToSpeech.JavaClass.init(TAndroidHelper.Context, FTTSOnInitListener);
{$ENDIF}
end;

procedure TAndroidTTS.Speak(say: String);
{$IFDEF ANDROID}
var
  Params: JHashMap;
begin
  Params := nil;
// This needs to be a <String,String> hashmap for the OnDone to work.
  Params := TJHashMap.JavaClass.init();
  Params.put(TJTextToSpeech_Engine.JavaClass.KEY_PARAM_UTTERANCE_ID,
     StringToJString('ID'));
  FTTS.speak(StringToJString(say), TJTextToSpeech.JavaClass.QUEUE_FLUSH, Params);
end;
{$ELSE}
begin

end;
{$ENDIF}

{$IFDEF ANDROID}
{ TAndroidTTS.TTTSOnInitListener }

constructor TAndroidTTS.TTTSOnInitListener.Create(AParent: TAndroidTTS);
begin
  inherited Create;
  FParent := AParent;
end;

procedure TAndroidTTS.TTTSOnInitListener.onInit(status: Integer);
var
  Result : Integer;
begin
  Log.d('onInit+');
  if status = TJTextToSpeech.JavaClass.SUCCESS then
  begin
    result := FParent.FTTS.setLanguage(TJLocale.JavaClass.UK);
    if (result = TJTextToSpeech.JavaClass.LANG_MISSING_DATA) or
       (result = TJTextToSpeech.JavaClass.LANG_NOT_SUPPORTED) then
      raise Exception.Create('This Language is not supported');
    // Processing after Init
    FParent.FTTS.setOnUtteranceCompletedListener(FParent.FTTSOnUtteranceCompletedListener);
  end
  else
    raise Exception.Create('Initialisation Failed!');
  Log.d('onInit-');
end;

constructor TAndroidTTS.TTTSOnUtteranceCompletedListener.Create(AParent: TAndroidTTS);
begin
  inherited Create;
  FParent := AParent;
end;

procedure TAndroidTTS.TTTSOnUtteranceCompletedListener.onUtteranceCompleted(
  utteranceID: JString);
begin
  Log.d('onUtteranceCompleted+');
  // Currently not firing.
  if Assigned(FParent.FDone) then
    FParent.FDone(FParent);
  Log.d('onUtteranceCompleted-');
end;

{$ENDIF}

end.